
/**
 * EMemory.h - (c) EliteMMO Network, 2015
 *
 * This unpublished material is proprietary to EliteMMO Network.
 * All rights reserved.
 *
 * The methods and techniques described herein are considered trade secrets
 * and/or condifential. Reproduction or distribution, in whole or in part,
 * is forbidden except by express written permission of EliteMMO Network.
 *
 * This file is part of the EliteMMO Development Kit.
 */

#ifndef __EDK_MEMORY_H_INCLUDED__
#define __EDK_MEMORY_H_INCLUDED__

#if defined (_MSC_VER) && (_MSC_VER >= 1020)
#pragma once
#endif

#include <Windows.h>
#include <algorithm>
#include <vector>

namespace EliteMMO
{
    namespace Memory
    {
        /**
         * @brief Scans a given chunk of data for the given pattern and mask.
         *
         * @param data          The data to scan within for the given pattern.
         * @param baseAddress   The base address of where the scan data is from.
         * @param lpPattern     The pattern to scan for.
         * @param pszMask       The mask to compare against for wildcards.
         * @param offset        The offset to add to the pointer.
         * @param resultUsage   The result offset to use when locating signatures that match multiple functions.
         *
         * @return Pointer of the pattern found, 0 otherwise.
         */
        static DWORD __stdcall FindPattern(std::vector<unsigned char> data, unsigned int baseAddress, const unsigned char* lpPattern, const char* pszMask, int offset, int resultUsage)
        {
            // Build vectored pattern..
            std::vector<std::pair<unsigned char, bool>> pattern;
            for (size_t x = 0; x < strlen(pszMask); x++)
                pattern.push_back(std::make_pair(lpPattern[x], pszMask[x] == 'x'));

            // The result count for multiple results..
            auto resultCount = 0;
            auto scanStart = data.begin();

            while (true)
            {
                // Search for the pattern..
                auto ret = std::search(scanStart, data.end(), pattern.begin(), pattern.end(),
                    [&](unsigned char curr, std::pair<unsigned char, bool> currPattern)
                {
                    return (!currPattern.second) || curr == currPattern.first;
                });

                // Did we find a match..
                if (ret != data.end())
                {
                    // If we hit the usage count, return the result..
                    if (resultCount == resultUsage || resultUsage == 0)
                        return (std::distance(data.begin(), ret) + baseAddress) + offset;

                    // Increment the found count and scan again..
                    ++resultCount;
                    scanStart = ++ret;
                }
                else
                    break;
            }

            return 0;
        }

        /**
         * @brief Determines the calling module based on the given return address.
         *
         * @param dwReturnAddress   The address being returned to from the call.
         *
         * @return Module base address on success, otherwise dwReturnAddress;
         */
        static HMODULE __stdcall GetCallingModule(DWORD dwReturnAddress)
        {
            if (dwReturnAddress != 0)
            {
                MEMORY_BASIC_INFORMATION mbi = { 0 };
                if (::VirtualQuery((LPCVOID)dwReturnAddress, &mbi, sizeof(MEMORY_BASIC_INFORMATION)) == sizeof(MEMORY_BASIC_INFORMATION))
                    return (HMODULE)mbi.AllocationBase;
            }
            return (HMODULE)dwReturnAddress;
        }
    }; // namespace Memory
}; // namespace EliteMMO

#endif // __EDK_MEMORY_H_INCLUDED__
